<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Carbon;

use App\Models\BankList;
use App\Models\BankDetail;
use App\Models\DispatchSet;
use App\Models\CheckList;
use App\Models\CheckBankDetail;

class DispatchTask extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'dispatch:task';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Command description';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        DB::beginTransaction();
        try {
            // 排程邏輯如下
            /*
                1. 抓取 Dispatch表中的 Cycle 和 endDate 和 state
                2. 當該 state 狀態 不為 0(失效) 時，才進行檢查派工及週期的動作
                    2.1 => 判斷 Cycle(週期) ， 對endDate做週期性的list巡檢單建置
                3. 檢核表
                4. 當巡檢單進行建置時， 對Bank_detail進行題目資料建置
            */
            $now = Carbon::now()->startOfDay();
            $dispatches = DispatchSet::where('state', '!=' , 0)->get();
            foreach ($dispatches as $dispatch) {
                $nextDate = Carbon::parse($dispatch->nextdate)->startOfDay();
                $eq_id = BankList::with('equipment')->where('id' , $dispatch->bank_id)->value('eq_id');

                if($nextDate->eq($now)) {
                    \Log::info("準備進行自動派工的派工單：ID={$dispatch->id}, 週期={$dispatch->cycle}");

                    //巡檢單 - 單建置
                    $newCheckList = CheckList::create([
                        'checker_id' => $dispatch->checker_id,
                        'eq_id' => $eq_id,
                        //假設 期限原本是 7/9~7/10 (週期：日) => 創建時應該會變成 startdate會是7/10 ，enddate會是7/11
                        'startdate' => $dispatch->nextdate,
                        'enddate' => calculateNextDate($dispatch->nextdate, $dispatch->cycle),
                        'cycle' => $dispatch->cycle,
                        'state' => 1,
                        'finish_user_id' => null,
                        'creator' => $dispatch->creator,
                    ]);

                    //巡檢單 - 題目建置
                    $bankData = BankDetail::where('bank_id',$dispatch->bank_id)->get();
                    foreach ($bankData as $data) {
                        CheckBankDetail::create([
                            'check_id' => $newCheckList->id,
                            'code' => $data->code,
                            'item' => $data->item,
                            'range' => $data->range,
                            'range2' => $data->range2,
                            'unit' => $data->unit,
                            'remark' => $data->remark,
                            'risk' => $data->risk,
                        ]);     
                    }

                    // $nextDate = match ($dispatch->cycle) {
                    //     'daily'  => $now->copy()->addDay(),
                    //     'week'   => $now->copy()->addWeek(),
                    //     'month'  => $now->copy()->addMonth(),
                    //     'season' => $now->copy()->addMonthsNoOverflow(3),
                    //     'year'   => $now->copy()->addYearNoOverflow(),
                    //     default  => null,
                    // };
                    //更新派工單下一筆建置的日期
                    $dispatch->update(['nextdate' => calculateNextDate($dispatch->nextdate, $dispatch->cycle)]);


                    DB::commit();
                    \Log::info("完成自動派工的派工單：ID={$dispatch->id}");
                }else{
                    \Log::info("時間尚未到的派工單：ID{$dispatch->id}");
                }
            }
        } catch (\Exception $e) {
            DB::rollback();
            \Log::error('任務失敗：' . $e->getMessage());
        }
    }
}

function calculateNextDate(Carbon $date, string $cycle): Carbon
{
    return match ($cycle) {
        'daily'   => $date->copy()->addDay(),
        'week'    => $date->copy()->addWeek(),
        'month'   => $date->copy()->addMonthNoOverflow(),
        'season'  => $date->copy()->addMonthsNoOverflow(3),
        'year'    => $date->copy()->addYearNoOverflow(),
        default   => throw new Exception("無效的週期類型：{$cycle}"),
    };
}
