<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Factory;
use App\Models\Equipment;
use App\Models\EquipmentCategory;
use App\Models\EquipmentRecord;
use App\Models\FactoryUser;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;

use Exception;

class EquipmentController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $result =  Equipment::with(['category','factory'])->where('creator', Auth::id())->orderBy('id')->get();

        return response()->json(['data' => $result], 200);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        DB::beginTransaction();
        try {
            $equipment = Equipment::create([
                'code' => $request->code,
                'eq_cat_id' => $request->eq_cat_id,
                'factory_id' => $request->factory_id,
                'name' => $request->name,
                'department' => $request->department,
                'remark' => $request->remark,
                'creator' => Auth::user()->id,
            ]);

            // 寫入異動紀錄(新增)
            EquipmentRecord::create([
                'eq_id' => $equipment->id,
                'code' => $equipment->code,
                'name' => $equipment->name,
                'eq_cat_id' => $equipment->eq_cat_id,
                'factory_id' => $equipment->factory_id,
                'department' => $equipment->department,
                'remark' => $equipment->remark,
                'state' => 1,
                'creator' => $equipment->creator,
            ]);

            DB::commit();
        } catch (Exception $e) {
            DB::rollback();
            throw $e;
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Equipment $equipment)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        DB::beginTransaction();
        try {
            $equipment = Equipment::findOrFail($id);
            $equipment->update([
                'code' => $request->code,
                'eq_cat_id' => $request->eq_cat_id,
                'factory_id' => $request->factory_id,
                'name' => $request->name,
                'department' => $request->department,
                'remark' => $request->remark,
            ]);

            // 寫入異動紀錄(修改)
            EquipmentRecord::create([
                'eq_id' => $equipment->id,
                'code' => $equipment->code,
                'name' => $equipment->name,
                'eq_cat_id' => $equipment->eq_cat_id,
                'factory_id' => $equipment->factory_id,
                'department' => $equipment->department,
                'remark' => $equipment->remark,
                'state' => 2,
                'creator' => $equipment->creator,
            ]);

            DB::commit();
        } catch (Exception $e) {
            DB::rollback();
            throw $e;
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Equipment $equipment)
    {
        DB::beginTransaction();
        try {
            $equipment->load('bankLists');
            if ($equipment->bankLists && $equipment->bankLists->count() > 0) {
                return response()->json(["message" => "該設備下有建置檢核表單，請先將檢核表單刪除"], 400);
            }
            $equipment->delete();

            // 寫入異動紀錄(刪除)
            EquipmentRecord::create([
                'eq_id' => $equipment->id,
                'code' => $equipment->code,
                'eq_cat_id' => $equipment->eq_cat_id,
                'factory_id' => $equipment->factory_id,
                'name' => $equipment->name,
                'remark' => $equipment->remark,
                'department' => $equipment->department,
                'state' => 0,
                'creator' => $equipment->creator,
            ]);

            DB::commit();
        } catch (Exception $e) {
            DB::rollback();
            throw $e;
        }
        
    }

    public function importEquipment(Request $request)
    {
        DB::beginTransaction();
        try {
            $file = $request->file('BankFile');
            $spreadsheet = IOFactory::load($file->getPathname());
            $worksheet = $spreadsheet->getActiveSheet();
            for($i=0;$i<$spreadsheet->getSheetCount();$i++){
                $worksheet = $spreadsheet->getSheet($i);
                $highestRow = $worksheet->getHighestRow(); // 總行數
                $highestColumn = $worksheet->getHighestColumn(); // 總列數
                $highestColumnIndex = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($highestColumn); // e.g. 5
                $nextHighestColumn = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($highestColumnIndex + 1);
        
                $lines = $highestRow - 2;
                if ($lines <= 0) {
                    return response()->json(['error' => 'Excel表格中沒有資料，請根據範例調整'], 400);
                }
        
                //============開始處理，設備的inform內容============
                $lastRow = $worksheet->getHighestDataRow('A'); // 總行數
                $dataArray = $worksheet->rangeToArray(
                  'A2:F' . $lastRow,    // The worksheet range that we want to retrieve
                  NULL,        // Value that should be returned for empty cells
                  TRUE,        // Should formulas be calculated (the equivalent of getCalculatedValue() for each cell)
                  TRUE,        // Should values be formatted (the equivalent of getFormattedValue() for each cell)
                  TRUE         // Should the array be indexed by cell row and cell column
                );
                foreach ($dataArray as $key => $value) {
                    $cate = EquipmentCategory::where('name', $value['B'])->first();
                    if (!$cate) {
                        return response()->json(['error' => '設備類別不存在，請確認是否輸入正確'], 400);
                    }
                    $factory = Factory::where('name',$value['D'])->first();
                    if(!$factory) {
                        return response()->json(['error' => '廠區不存在，請確認是否輸入正確'], 400);
                    }
                    $equipment = Equipment::create([
                        'code' => $value['A'],
                        'eq_cat_id' => $cate->id,
                        'name' => $value['C'],
                        'factory_id' => $factory->id,
                        'department' => $value['E'],
                        'remark' => $value['F'],
                        'creator' => Auth::id(),
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);

                    EquipmentRecord::create([
                        'eq_id' => $equipment->id,
                        'code' => $value['A'],
                        'name' => $value['C'],
                        'eq_cat_id' => $cate->id,
                        'factory_id' => $factory->id,
                        'department' => $value['E'],
                        'remark' => $value['F'],
                        'state' => 1,
                        'creator' => Auth::id(),
                    ]);
                }
                //=================================================
              }
            DB::commit();
        } catch (Exception $e) {
            DB::rollback();
            throw $e;
        }
    }

     public function ExportEquipment(Request $request){
        DB::beginTransaction();
        try {  
            $result = Equipment::with(['category','factory'])->where("creator", Auth::user()->id)->where('deleted_at','=',null)->get();
            // 匯出設置
            $path = public_path("export.xlsx");
            $spreadsheet = IOFactory::load($path);
            $styleArray = [
                'font' => [
                    'bold' => true,
                ],
                'alignment' => [
                    'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER,
                    'vertical' => \PhpOffice\PhpSpreadsheet\Style\Alignment::VERTICAL_CENTER,
                    'indent' => 1,
                    'wrapText' => true,
                ],
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                        'color' => ['argb' => '000000'],
                    ],
                ],
                'fill' => [
                  'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                  'startColor' => ['rgb' => 'DCE6F1'],
                ],
              ];

            $worksheet = $spreadsheet->getSheetByName('Excel');
            $spreadsheet->getActiveSheet()->setCellValue("A1", "設備編號");
            $spreadsheet->getActiveSheet()->setCellValue("B1", "設備類別");
            $spreadsheet->getActiveSheet()->setCellValue("C1", "設備名稱");
            $spreadsheet->getActiveSheet()->setCellValue("D1", "廠區");
            $spreadsheet->getActiveSheet()->setCellValue("E1", "部門");
            $spreadsheet->getActiveSheet()->setCellValue("F1", "設備位置及備註"); 
            $spreadsheet->getActiveSheet()->getStyle('A1:F2')->applyFromArray($styleArray);
            $columns = ["A","B","C","D","E","F"];
            foreach ($columns as $column) {
              $worksheet->getColumnDimension($column)->setWidth(20);
              $spreadsheet->getActiveSheet()->mergeCells($column."1:".$column."2");  
            }

            //用於定向當前該寫入資料的位置
            $rowCount = 3;
            $startCateRow = $rowCount; // 主項起始行

            foreach ($result as $value) {
                $spreadsheet->getActiveSheet()->setCellValue("A". $rowCount , $value->code);
                $spreadsheet->getActiveSheet()->setCellValue("B". $rowCount , $value->category->name);
                $spreadsheet->getActiveSheet()->setCellValue("C". $rowCount , $value->name);
                $spreadsheet->getActiveSheet()->setCellValue("D". $rowCount , $value->factory->name);
                $spreadsheet->getActiveSheet()->setCellValue("E". $rowCount , $value->department);
                $spreadsheet->getActiveSheet()->setCellValue("F". $rowCount , $value->remark);

                $rowCount++;
            }

            $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
            ob_start();
            $writer->save('php://output');
            $ExcelExport = ob_get_clean();
            $ExportExcelEncode = base64_encode($ExcelExport);
            $response = array(
                'name' => "exportBank",
                'file' => "data:application/vnd.openxmlformats-officedocument.spreadsheetml.sheet;base64,".$ExportExcelEncode
            );
            return response()->json($response);
        } catch (Exception $e) {
            DB::rollback();
            throw $e;
        }
    }

}