<?php

namespace App\Http\Repositories;

use App\Helpers\UserHelper;
use App\Models\ApprovalDoc;
use App\Models\ApprovalDocFile;
use App\Models\ApprovalDocList;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\Storage;

class ApprovalDocRepository
{

    function __construct(protected ApprovalDoc $doc, protected ApprovalDocList $doc_list, protected ApprovalDocFile $doc_file) {}

    //列出全部的核可書，計算列表總數
    public function all(): Collection
    {
        $company_id = UserHelper::getCompanyId();
        return $this->doc->with('factory')
            ->withCount('docList')
            ->where('company_id', $company_id)->get();
    }
    //找出核可書及明細
    public function docfind($id)
    {
        return $this->doc->with([
            'docList' => function ($query) {
                $query->orderBy('created_at', 'asc')  // 對 docList 進行排序
                    ->with('chemical:id,ch_name,listNo,toxic_class', 'factory:id,name');
            },
            'docFile'
            ,'factory:id,venderNo'
        ])->find($id);

        if ($doc && $doc->docFile) {
            // 如果 docFile 是一個集合，則遍歷並修改 file_path
            $doc->docFile->each(function ($file) {
                $file->file_path = config('app.asset_url') . Storage::url($file->file_path);
            });
        }
        return $doc;
    }
    //找出單一核可證書
    public function docOnlyFind($id)
    {
        return $this->doc->find($id);
    }
    //找出可用證書
    public function docNow()
    {
        $company_id = UserHelper::getCompanyId();
        return $this->doc->join('approval_doc_lists as l','l.doc_id','=','approval_docs.id')
        ->join('factories as f','f.id','=','approval_docs.factory_id')
        // ->with('factory')
        ->where('l.company_id',$company_id)
        // ->where('end_date','>',now())
        ->get();
    }
    //使用化學品id與時間查詢核可證書
    public function chemFindDoc($id,$add_date)
    {
        $company_id = UserHelper::getCompanyId();
        $dd = $this->doc
        ->rightjoin('approval_doc_lists as l','l.doc_id','approval_docs.id')
        ->where('end_date','>',$add_date)
        ->where('start_date','<',$add_date)
        ->where('chemical_id',$id)
        ->where('l.company_id',$company_id)
        //相減取最高
        ->orderByRaw('(conc_up::double precision - conc_low::double precision) desc')
        ->limit(1);
        return $dd->first();
    }
    public function manyChemFindDoc($id)
    {
        $company_id = UserHelper::getCompanyId();
        $dd = $this->doc
        ->rightjoin('approval_doc_lists as l','l.doc_id','approval_docs.id')
        ->where('end_date','>',now())
        ->where('chemical_id',$id)
        ->where('l.company_id',$company_id);
        return $dd->get();
    }
    //使用濃度查詢
    public function chemConcFindDoc($id,$conc)
    {
        $company_id = UserHelper::getCompanyId();
        $dd = $this->doc
        ->rightjoin('approval_doc_lists as l','l.doc_id','approval_docs.id')
        ->whereRaw('CAST(conc_low AS DECIMAL) <= ?', [$conc])
        ->whereRaw('CAST(conc_up AS DECIMAL) >= ?', [$conc])
        ->where('end_date','>',now())
        ->where('chemical_id',$id)
        ->where('l.company_id',$company_id);
        return $dd->get();
    }
    //使用運作管制編號查詢
    public function chemFactoryFindDoc($id,$conc,$factory,$operate_date)
    {
        $company_id = UserHelper::getCompanyId();
        $dd = $this->doc
        ->rightjoin('approval_doc_lists as l','l.doc_id','approval_docs.id')
        ->whereRaw('CAST(conc_low AS DECIMAL) <= ?', [$conc])
        ->whereRaw('CAST(conc_up AS DECIMAL) >= ?', [$conc])
        // ->where('factory_id',$factory)
        ->where('end_date','>=',$operate_date)
        ->where('chemical_id',$id)
        ->where('l.company_id',$company_id);
        return $dd->first();
    }

    //建立核可書
    public function docCreate($data)
    {
        return $this->doc->create($data)->id;
    }
    //更新核可書
    public function docUpdate($id, $data)
    {
        $hasdoc = $this->doc->find($id);
        if ($hasdoc) {
            $updated = $hasdoc->update($data);
            return $updated ? $hasdoc->id : null;
        }
    }
    //刪除證書
    public function docDelete($id)
    {
        return $this->doc->destroy($id);
    }
    //找出單一證書列表
    public function docListGet($id)
    {
        return $this->doc_list->where('doc_id', $id)->get();
    }
    //找出單一證書單一化學品
    public function docListfind($doc_id,$chemical_id){
        return $this->doc_list->where('doc_id',$doc_id)
        ->where('chemical_id',$chemical_id)->get();
    }
    //列表建立
    public function docListCreate($data)
    {
        return $this->doc_list->create($data);
    }
    //列表更新
    public function docListUpdate($id, $data)
    {
        $hasList = $this->doc_list->find($id);
        if ($hasList) {
            return $hasList->update($data);
        }
    }
    //列表刪除
    public function docListDelete($id)
    {
        return $this->doc_list->destroy($id);
    }
    //找出核可書的上傳檔案
    public function docFileFind($id)
    {
        return $this->doc_file->where('doc_id', $id)->get();
    }
    //上傳檔案建立
    public function docFileCreate($data)
    {
        return $this->doc_file->create($data);
    }
    //上傳檔案更新
    public function docFileupdate($data)
    {
        return $this->doc_file->update($data);
    }
    //刪除檔案
    public function deleteDocFile($id)
    {
        return $this->doc_file->destroy($id);
    }
}
