<?php

namespace App\Http\Services;

use App\Helpers\UserHelper;
use App\Http\Repositories\ApprovalDocRepository;
use App\Http\Repositories\ChemicalBaseRepository;
use App\Http\Repositories\FactoryRepositories;
use App\Models\ApprovalDoc;
use Exception;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class ApprovalDocService
{

    function __construct(protected FactoryRepositories $factory, protected ChemicalBaseRepository $chem, protected ApprovalDocRepository $approval) {}

    public function getSelectList()
    {
        $factories = $this->factory->getFactoryAll();
        $chemical = $this->chem->getAllWithListNo();
        return ['factory' => $factories, 'chemical' => $chemical];
    }

    public function all()
    {
        return $this->approval->all();
    }
    public function docfind($id)
    {
        return $this->approval->docfind($id);
    }
    public function docCreate($request)
    {
        DB::beginTransaction();
        try {
            $user = Auth::user();
            $company_id = UserHelper::getCompanyId();
            $filePath = null;
            $fileName = null;
            $fileDataList = [];
            if ($request->hasFile('file')) {
                foreach ($request->file('file') as $file) {
                    $fileName = $file->getClientOriginalName();
                    $parts = pathinfo($fileName);
                    $extension = isset($parts['extension']) ? strtolower($parts['extension']) : '';

                    if (!in_array($extension, ['pdf', 'docx'])) {
                        throw new \Exception('不支援的檔案類型');
                    }

                    $saveFileName = $parts['filename'] . '_' . date('YmdHi') . '_' . Str::random(6) . '.' . $extension;
                    $filePath = $file->storeAs('public/ApprovalDoc', $saveFileName);

                    if (!$filePath) {
                        throw new \Exception('檔案儲存失敗');
                    }
                    $fileDataList[] = [
                        'file_name' => $fileName,
                        'file_path' => config('app.asset_url') . Storage::url($filePath)
                    ];
                }
            }
            if (empty($request->doc_id)) {
                $data = [
                    'per_no' => $request->per_no,
                    'factory_id' => $request->factory_id,
                    'people' => $request->people,
                    'start_date' => $request->start_date,
                    'end_date' => $request->end_date,
                    'company_id' => $company_id,
                    'created_user' => $user->id,
                    'updated_user' => $user->id,
                ];
                $result = $this->approval->docCreate($data);
                foreach ($fileDataList as $fileData) {
                    $fileData['doc_id'] = $result;
                    $this->approval->docFileCreate($fileData);
                }
            } else {
                $id = $request->doc_id;
                $existingFiles  = $this->approval->docFileFind($id);
                $data = [
                    'factory_id'=>$request->factory_id,
                    'start_date'=>$request->start_date,
                    'end_date'=>$request->end_date,
                    'people'=>$request->people,
                    'per_no' => $request->per_no,
                    'updated_user' => $user->id,
                ];
                if (!empty($fileDataList)) {
                    // 刪除舊檔案
                    foreach ($existingFiles as $file) {
                        if (!empty($file->file_path) && Storage::exists($file->file_path)) {
                            Storage::delete($file->file_path);
                        }
                        $this->approval->deleteDocFile($file->id); // 刪除資料庫記錄
                    }
                    foreach ($fileDataList as $fileData) {
                        $fileData['doc_id'] = $id;
                        $this->approval->docFileCreate($fileData);
                    }
                }
                $result = $this->approval->docUpdate($id, $data);
            }
            DB::commit();
            return $result;
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }
    public function docDelete($ids)
    {
        DB::beginTransaction();
        try {
            foreach ($ids as $id) {
                $docs = $this->approval->docListGet($id);
                $docs->each->delete();
                $existingFiles  = $this->approval->docFileFind($id);
                    foreach ($existingFiles as $file) {
                        if (!empty($file->file_path) && Storage::exists($file->file_path)) {
                            Storage::delete($file->file_path);
                        }
                        $this->approval->deleteDocFile($file->id); // 刪除資料庫記錄
                    }
            }
            $result = $this->approval->docDelete($ids);
            DB::commit();
            return $result;
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }
    public function docListCreate($request)
    {
        if(!$request->doc_id){
            throw new Exception("未建立核可文件");
        }
        if($request->conc_low > $request->conc_up )
        {
            throw new Exception("濃度下限不得大於濃度上限");
        }
        $checkRepeat = $this->approval->docListfind($request->doc_id,$request->chemical_id);
        if($checkRepeat)
        {
            foreach($checkRepeat as $repeat){
                if(!($request->conc_up < $repeat->conc_low || $request->conc_low > $repeat->conc_up)){
                    throw new Exception("化學品濃度範圍重疊");
                }
            }
        }
        DB::beginTransaction();
        try {
            $user = Auth::user();
            $company_id = UserHelper::getCompanyId();
            $data = array_merge(
                $this->changeType($request),
                $request->toArray(),
                [
                    'company_id' => $company_id,
                    'created_user' => $user->id,
                    'updated_user' => $user->id,
                ]
            );
            $this->approval->docListCreate($data);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    public function changeType($type){
        $data = [];
        $data['type_storage']=false;
        $data['type_use'] = false;
        $data['type_make'] = false;
        $data['type_shift_in'] = false;
        $data['type_shift_out'] = false;
        $data['type_sell'] = false;
        $data['type_export'] = false;
        $data['type_import'] = false;
        if($type->type_storage){
            $data['type_storage']= true;
        }
        if($type->type_use)
        {
            $data['type_use'] = true;
        }
        if($type->type_make)
        {
            $data['type_make'] = true;
        }
        if($type->type_import)
        {
            $data['type_import'] = true;
        }
        if($type->type_export)
        {
            $data['type_export'] = true;
        }
        if($type->type_sell)
        {
            $data['type_sell'] = true;
            $data['type_shift_in'] = true;
            $data['type_shift_out'] = true;
        }
        $data['type_buy'] = true;
        $data['type_decant'] = true;
        $data['type_transfer'] = true;
        $data['type_disposal'] = true;
        $data['type_residual'] = true;
        $data['type_inventory_plus'] = true;

        return $data;
    }

    public function docListUpdate($request)
    {
        DB::beginTransaction();
        try {
            $user = Auth::user();
            $data = array_merge(
                $this->changeType($request),
                $request->toArray(),
                [
                    'updated_user' => $user->id
                ]
            );
            $this->approval->docListUpdate($request->id, $data);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }
    public function docListDelete($id)
    {
        return $this->approval->docListDelete($id);
    }

    public function manyChemFindDoc($id)
    {
        return $this->approval->manyChemFindDoc($id);
    }
}
