<?php

namespace App\Http\Services;

use App\Http\Repositories\ChemicalOperationRepository;
use Carbon\Carbon;

class StatisticsService
{
    function __construct(protected ChemicalOperationRepository $chem_op) {}


    function getGraphData($user, $company, $request){
        // 第一個查詢：獲取所有操作日期的年份，並聚合
        // 這裡的 get() 會返回一個 Collection
        $year = $this->chem_op->getGraphYear($user,$company)->selectRaw('EXTRACT(YEAR FROM chemical_operations.operate_date) as year')->groupby('year')->get();
    
        // 第二個查詢：獲取指定年份的詳細資料
        // 這裡的 get() 會返回一個 Collection
        $data = $this->chem_op->getGraphData($user,$company)->whereYear('chemical_operations.operate_date', $request->year)->orderby('chemical_operations.operate_date','asc')->get();
    
        $daily = [];
        $weekly = [];
        $monthly = [];
        $quarterly = [];
    
        // 初始化指定年份的所有月份和季度的數據為0
        $startDate = Carbon::createFromDate($request->year, 1, 1);
        $endDate = Carbon::createFromDate($request->year, 12, 31);
    
        // 初始化每日數據
        for ($date = $startDate->copy(); $date <= $endDate; $date->addDay()) {
            $dayKey = $date->format('Y-m-d');
            $daily[$dayKey] = 0;
        }
    
        // 初始化每週數據：使用更穩健的初始化方法
        // 遍歷當前年份的每個月份，並為該月中的每個可能週鍵初始化
        $currentMonthIterator = $startDate->copy();
        while ($currentMonthIterator->year == $request->year) {
            for ($day = 1; $day <= $currentMonthIterator->daysInMonth; $day++) {
                $date = Carbon::createFromDate($request->year, $currentMonthIterator->month, $day);
                $weekOfMonth = (int) ceil($date->day / 7);
                $weekKey = $date->format('Y-m') . '-W' . $weekOfMonth;
                if (!isset($weekly[$weekKey])) { // 避免重複設定，雖然結果相同
                    $weekly[$weekKey] = 0;
                }
            }
            $currentMonthIterator->addMonth();
        }
    
        // 初始化月份數據
        for ($date = $startDate->copy(); $date <= $endDate; $date->addMonth()) {
            $monthKey = $date->format('Y-m');
            $monthly[$monthKey] = 0;
        }
    
        // 初始化季度數據
        for ($date = $startDate->copy(); $date <= $endDate; $date->addQuarter()) {
            $quarter = (int) ceil($date->month / 3);
            $quarterKey = $date->year . '-Q' . $quarter;
            $quarterly[$quarterKey] = 0;
        }
        foreach ($data as $item) {
            $date = Carbon::parse($item->operate_date);
            $weight = (float) $item->op_weight; // 確保 weight 是數值型態
    
            // 每日
            $dayKey = $date->format('Y-m-d');
            $daily[$dayKey] += $weight; // 使用 += 更簡潔
    
            // 每週（以月份的第幾週為準）
            $weekOfMonth = (int) ceil($date->day / 7);
            $weekKey = $date->format('Y-m') . '-W' . $weekOfMonth;
            $weekly[$weekKey] += $weight;
    
            // 每月
            $monthKey = $date->format('Y-m');
            $monthly[$monthKey] += $weight;
    
            // 每季
            $quarter = (int) ceil($date->month / 3);
            $quarterKey = $date->year . '-Q' . $quarter;
            $quarterly[$quarterKey] += $weight;
        }
    
        return [
            'year' => $year, // 這裡的 $year 已經是 Collection，可以直接返回
            // 為了確保輸出順序，可以加上 sortKeys()
            'daily' => collect($daily)->sortKeys()->map(fn($weight, $date) => ['date' => $date, 'weight' => $weight])->values(),
            'weekly' => collect($weekly)->sortKeys()->map(fn($weight, $week) => ['week' => $week, 'weight' => $weight])->values(),
            'monthly' => collect($monthly)->sortKeys()->map(fn($weight, $month) => ['month' => $month, 'weight' => $weight])->values(),
            'quarterly' => collect($quarterly)->sortKeys()->map(fn($weight, $quarter) => ['quarter' => $quarter, 'weight' => $weight])->values(),
        ];
    }

    function getOperateData($user, $company, $request){
            $query = $this->chem_op->getOperateData($user, $company,$request);
            return $query->get();
    }

}