<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Services\UserService;
use App\Services\SubSystemService;
use App\Models\SubSystem;
use App\Models\User;
use App\Models\UserPlan;
use App\Models\UserPlanItem;
use App\Models\UserPlanRecord;

class SubSystemController extends Controller
{
    private $UserService;
    private $SubSystemService;

    public function __construct(
        UserService $UserService,
        SubSystemService $SubSystemService
    )
    {
        $this->UserService = $UserService;
        $this->SubSystemService = $SubSystemService;
    }

    public function getSubSystemWithItem()
    {
        $result = SubSystem::with('items')->where('type', 1)->get();

        return response()->json(['data' => $result], 200);
    }

    public function getSubSystemWithPlan()
    {
        $result = SubSystem::with('plans.items.item')->where('type', 1)->get();

        return response()->json(['data' => $result], 200);
    }

    public function getAllUserSubSystemWithPlan()
    {
        /*
        SELECT 
            users.name, 
            sub_systems.name, 
            userplan.*
        FROM 
            users
        CROSS JOIN 
            sub_systems
        LEFT JOIN (
            SELECT 
                UP.*, 
                P.sub_system_id, 
                ROW_NUMBER() OVER (PARTITION BY UP.user_id, P.sub_system_id ORDER BY UP.created_at DESC) AS rn
            FROM 
                user_plans UP
            INNER JOIN 
                plans P ON P.id = UP.plan_id
        ) AS userplan 
        ON 
            userplan.user_id = users.id 
            AND userplan.sub_system_id = sub_systems.id 
            AND userplan.rn = 1
        WHERE 
            users.permission = '100';
        */
        
        // 建立子查詢
        $subQuery = UserPlan::select('user_plans.*')
            ->selectRaw('ROW_NUMBER() OVER (PARTITION BY user_id, sub_system_id 
                ORDER BY CASE state
                    WHEN 1 THEN 0
                    WHEN 0 THEN 1
                    WHEN 2 THEN 2
                    WHEN 3 THEN 3
                END, 
                created_at DESC) AS rn');
            
        // 主查詢
        $result = User::select('users.id', 'users.name as user_name', 'users.email as user_email', 'sub_systems.id as sub_system_id', 'sub_systems.name as sub_system_name', 'userplan.plan_name', 'userplan.state', 'userplan.open_date', 'userplan.expiry_date')
            ->crossJoin('sub_systems')
            ->leftJoinSub($subQuery, 'userplan', function ($join) {
                $join->on('userplan.user_id', '=', 'users.id')
                    ->on('userplan.sub_system_id', '=', 'sub_systems.id')
                    ->where('userplan.rn', '=', 1);
            })
            ->where('users.permission', '100')
            ->where('sub_systems.type', 1)
            ->get();

        // 整理格式
        $grouped = $result->groupBy('user_name')
            ->map(function ($group) {
                return [
                    'id' => $group[0]['id'],
                    'user_name' => $group[0]['user_name'],
                    'user_email' => $group[0]['user_email'],
                    'sub_systems' => $group->map(function ($item) {
                        return [
                            'sub_system_id' => $item['sub_system_id'],
                            'sub_system_name' => $item['sub_system_name'],
                            'plan_name' => $item['plan_name'],
                            'state' => $item['state'],
                            'open_date' => $item['open_date'],
                            'expiry_date' => $item['expiry_date']
                        ];
                    })->all()
                ];
            })->values()->all();
        
        return response()->json(['data' => $grouped], 200);
    }

    public function getOneUserSubSystemWithPlan($user_id)
    {
        $result = UserPlan::with('subsystem')->where('user_id', $user_id)->orderBy('created_at', 'desc')->get();
        $user = $this->UserService->show($user_id);
        
        return response()->json(['data' => $result, 'user' => $user], 200);
    }

    public function getOneUserSubSystemState($user_id)
    {
        $result = $this->SubSystemService->getOneUserSubSystemState($user_id);
        return response()->json(['data' => $result], 200);
    }

    public function checkHasSubSystemPermision($user_id)
    {
        $user = $this->UserService->subSystemPermission($user_id);

        return response()->json(['data' => $user], 200);
    }

    public function getSubSystemGroups()
    {
        $result = SubSystem::with('groups')->get();

        return response()->json(['data' => $result], 200);
    }

    public function getSubSystemUsage()
    {
        $result = DB::table('sub_systems')
            ->leftJoin('user_plans', function($join) {
                $join->on('sub_systems.id', '=', 'user_plans.sub_system_id')
                     ->where('user_plans.state', '=', 1);
            })
            ->select('sub_systems.*', DB::raw('count(user_plans.id) as user_plan_count'))
            ->groupBy('sub_systems.id')
            ->orderBy('sub_systems.id')
            ->get();

        
        return response()->json(['data' => $result], 200);
    }

    public function getUserSubHasSystemRole($system_id, $type, $user_id)
    {
        $query = User::whereHas('systemRole', function ($query) use($system_id) {
            $query->where('sub_system_id', $system_id)->where('state', true);
        });
        switch($type) {
            case 'p':
                $result = $query->where('permission', '100')->get();
                break;
            case 's':
                $result = $query->where('belongs', $user_id)->where('permission', '010')->get();
                break;
            case 'u':
                $result = $query->where('belongs', $user_id)->where('permission', '001')->get();
                break;
        }

        // foreach ($result as $user) {
        //     $user->jwtToken = $user->jwtToken($system_id);
        // }

        $user = User::with("belongsParent")->where('id', $user_id)->first();

        // return response()->json(['data' => $user], 200);
        return response()->json(['data' => $result, 'subName' => $user->name, 'parentName' => $user->belongsParent->name ?? $user->name], 200);
    }

    public function getSubSystemWithStatistics(Request $request)
    {
        $startDate = $request->startDate;
        $endDate = $request->endDate;

        $subSystems = SubSystem::query()
            ->select('sub_systems.id', 'sub_systems.name')
            ->with(['plans' => function($query) use($startDate, $endDate) {
                $query->leftJoin('user_plans', function($join) use($startDate, $endDate) {
                    $join->on('plans.sub_system_id', '=', 'user_plans.sub_system_id')
                        ->on('plans.name', '=', 'user_plans.plan_name')
                        ->where('user_plans.state', '=', 1);

                    if (isset($startDate)) {
                        $join->where('user_plans.open_date', '>=', $startDate);
                    }
                    if (isset($endDate)) {
                        $join->where('user_plans.open_date', '<=', $endDate);
                    }
                })
                ->whereNull('plans.deleted_at')
                ->select('plans.*', DB::raw('count(user_plans.id) as active_num'))
                ->groupBy('plans.id', 'plans.name');
            }])
            ->orderBy('sub_systems.id')
            ->get();
        
        return response()->json(['data' => $subSystems], 200);
    }

    public function getSubSystemWithStatisticsDetail(Request $request)
    {
        $plan_id = $request->plan_id;
        $sub_system_id = $request->sub_system_id;

        $subSystems = SubSystem::query()
            ->select('sub_systems.id as sub_system_id', 'sub_systems.name as sub_system_name', 'plans.name as plan_name')
            ->join('plans', 'plans.sub_system_id', 'sub_systems.id')
            ->where('sub_systems.id', $sub_system_id)
            ->where('plans.id', $plan_id)
            ->first();

        $userPlans = UserPlan::query()
            ->where('sub_system_id', $subSystems->sub_system_id)
            ->where('plan_name', $subSystems->plan_name)
            ->where('state', 1)
            ->with('creator')
            ->get();

        $subSystems['user_plans'] = $userPlans;
        
        return response()->json(['data' => $subSystems], 200);
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $result = SubSystem::where('type', 1)->get();

        return response()->json(['data' => $result], 200);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     */
    public function show(SubSystem $subSystem)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(SubSystem $subSystem)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, SubSystem $subSystem)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(SubSystem $subSystem)
    {
        //
    }
}
