<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Auth\Events\Registered;
use Tymon\JWTAuth\Facades\JWTAuth;
use App\Models\User;
use App\Models\UserSystemRole;
use App\Models\PasswordLog;
use App\Services\UserService;
use App\Services\SSOService;
use Exception;
use Carbon\Carbon;

class UserController extends Controller
{
    private UserService $UserService;
    private SSOService $SSOService;

    public function __construct(UserService $UserService, SSOService $SSOService)
    {
        $this->UserService = $UserService;
        $this->SSOService = $SSOService;
    }

    function getUser(){
        if(Auth::check()){
            $user = Auth::user();
            
            $user->system_role = collect($this->UserService->getUserHasSubSystem($user->id))->map(function ($item) {
                return [
                    'sub_system' => [
                        'id' => $item['sub_system']->id ?? null,
                        'name' => $item['sub_system']->name ?? null,
                    ],
                    'state' => $item['state'],
                    'role' => $item['role'],
                ];
            })->toArray();

            return response()->json($user);
        }
        return response()->json(['error' => 'Unauthorized'], 401);
    }

    public function login(Request $request) {
        $validateUser = Validator::make($request->all(),
            [
                'email' => ['required', 'email'],
                'password' => ['required'],
            ]);

        if($validateUser->fails()){
            return response()->json([
                'status' => false,
                'message' => '驗證失敗！',
                'errors' => $validateUser->errors()
            ], 500);
        }

        if (Auth::attempt($request->only('email', 'password'))) {
            $user = Auth::user();
            // $user->api_token = Str::random(64);
            // $user->save();

            //非子帳號不能登入
            // if ($user->permission != '001') {
            //     return response()->json(['error' => "帳號或密碼錯誤"], 500);
            // }

            $tokens = $this->UserService->getSubsystemToken($user);
            $hotbee_token = $tokens['hotbee_token'];
            $polibee_token = $tokens['polibee_token'];

            $data = [
                'main_token' => $user->api_token,
                'hotbee_token' => $hotbee_token,
                'polibee_token' => $polibee_token,
            ];
            
            return response()->json(['success' => true, 'data' => $data]);
        } else {
            return response()->json(['error' => "帳號或密碼錯誤"], 500);
        }
    }

    public function logout(Request $request)
    {
        try {
            $user = Auth::user();
            $user->api_token = Str::random(64);
            $user->save();

            return response()->json(['success' => true]);
        } catch(Exception $e) {
            return response()->json(['error' => "Unauthorized."], 500);
        }
    }

    public function getJwtToken()
    {
        // $user = Auth::user();
        // $token = JWTAuth::claims(['account' => $user->account])->fromUser($user);

        $user = Auth::user();

        $systemRole = UserSystemRole::where('user_id', $user->id)->where('state', true)->get();

        $arr = [];
        foreach ($systemRole as $system) {
            $arr[$system->sub_system_id] = $user->jwtToken($system->sub_system_id);
        }
        
        return response()->json(['data' => $arr], 200);
    }

    public function getJwtTokenByUserId($id)
    {
        $user = User::find($id);

        $systemRole = UserSystemRole::where('user_id', $user->id)->where('state', true)->get();

        $arr = [];
        foreach ($systemRole as $system) {
            $arr[$system->sub_system_id] = $user->jwtToken($system->sub_system_id);
        }

        if ($user->permission == '999') {
            $arr[3] = $user->jwtToken(3);
        }
        
        return response()->json(['data' => $arr], 200);
    }

    public function getAllAdmins()
    {
        $result = $this->UserService->getAllAdmins();
        return response()->json(['data' => $result], 200);
    }

    public function getAllParents()
    {
        $result = $this->UserService->getAllParents();
        return response()->json(['data' => $result], 200);
    }

    public function getSubByParentId($parent_id)
    {
        if(Auth::user()->permission !== '999' && Auth::user()->permission !== '111' && Auth::user()->id != $parent_id) {
            return response()->json(['error' => 'Not authorized.'], 403);
        }
        $parent = $this->UserService->show($parent_id);
        return response()->json(['data' => $parent->sub, 'parentName' => $parent->name], 200);
    }

    public function getUserBySubId($sub_id)
    {
        if(Auth::user()->permission !== '999' && Auth::user()->permission !== '111' && Auth::user()->permission !== '100' && Auth::user()->id != $sub_id) {
            return response()->json(['error' => 'Not authorized.'], 403);
        }

        $sub = $this->UserService->show($sub_id);
        $data = $sub->subUser;
        // dd($sub->subUser);
        // if (Auth::user()->permission === '100') {
        //     $data = $sub->subUser;
        // }else{
        //     $data = $sub->sub;
        // }

        return response()->json(['data' => $data, 'subName' => $sub->name, 'parentName' => $sub->belongsParent->name ?? null], 200);
    }

    public function getParentsHasSubSystem($id)
    {
        $result = UserSystemRole::with('subSystem')->where('user_id', $id)->get();

        return response()->json(['data' => $result], 200);
    }

    public function getUserHasSubSystem($id)
    {
        $result = $this->UserService->getUserHasSubSystem($id);

        return response()->json(['data' => $result], 200);
    }

    public function userResetPassword(Request $request)
    {
        // TODO 變更密碼需要紀錄，變更時也需要先檢查與驗證，可能需要其他API來取得最後變更密碼時間。
        // 1. 密碼規則驗證(英文大小寫+數字)
        // 2. 密碼不得與前三次相同
        
        $user = User::find($request->id);
        // 檢查原密碼是否正確
        if (!Hash::check($request->old_password, $user->password)) {
            return 1;
        }

        // 檢查新密碼與確認新密碼是否一致
        if ($request->password != $request->password_confirmation) {
            return 2;
        }
        $rules = [
            'password' => [
                'required',
                'min:8', // 最少8個字元
                'regex:/[a-z]/', // 至少一個小寫字母
                'regex:/[A-Z]/', // 至少一個大寫字母
                'regex:/[0-9]/', // 至少一個數字
            ],
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return 3;
        }

        // 檢查新密碼是否與前3次相同
        $passwordHistories = $user->passwordLog()->take(3)->get();
        foreach($passwordHistories as $passwordHistory){
            if (Hash::check($request->password, $passwordHistory->password)) {
                return 4;
            }
        }

        // 加密新密碼，並更新
        $password = Hash::make($request->password);
        $user->update([
            'password' => $password,
        ]);

        // 寫入密碼變更紀錄
        PasswordLog::create([
            'user_id' => $user->id,
            'password' => $user->password
        ]);

        return 0;
    }

    public function getUserPermissionStatistics(Request $request)
    {
        $startDate = $request->startDate;
        $endDate = $request->endDate;

        $str = "
            select 
                permission, 
                case
                    when permission = '999' then '系統管理員'
                    when permission = '111' then '一般管理員'
                    when permission = '100' then '母公司'
                    when permission = '010' then '子公司'
                    when permission = '001' then '子帳號'
                end as name,
                count(permission) as num
            from users
                where deleted_at is null
        ";

        $params = [];
        if (isset($startDate)) {
            $str .= " and created_at >= :startDate";
            $params['startDate'] = $startDate;
        }
        if (isset($endDate)) {
            $str .= " and created_at <= :endDate";
            $params['endDate'] = $endDate;
        }
    
        $str .= " group by permission order by permission desc";

        $result = DB::select($str, $params);

        return response()->json(['data' => $result], 200);
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        // $this->authorize('isParent', User::class);

        $result = User::all();
        return response()->json(['data' => $result], 200);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        DB::beginTransaction();
        try {
            $messages = [
                'email.email' => 'Email格式不正確。',
                'email.unique' => 'Email無法使用。',
            ];

            //Validated
            $validateUser = Validator::make($request->all(),
            [
                'name' => ['required', 'string', 'max:255'],
                'email' => ['required', 'string', 'email', 'max:255', Rule::unique('users')->whereNull('deleted_at')],
            ], $messages);

            if($validateUser->fails()){
                return response()->json([
                    'status' => false,
                    'message' => '驗證錯誤',
                    'errors' => $validateUser->errors()
                ], 500);
            }

            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'permission' => $request->permission,
                'belongs' => $request->belongs ?? null,
                'group_id' => $request->group_id ?? null,
                'password' => Hash::make(Str::random(8)),
                'api_token' => Str::random(64),
                'uuid' => Str::random(64),
                'email_verified_at' => Carbon::now(),
            ], 201);

            $hasRole = false;
            $role = "";
            switch ($request->permission) {
                case '010':
                    $hasRole = true;
                    $role = 'sub';
                    break;
                case '001':
                    // 需要處理選擇角色的問題
                    $hasRole = true;
                    break;
            }

            if ($hasRole && is_array($request->items)) {
                foreach ($request->items as $item) {
                    UserSystemRole::create([
                        'user_id' => $user->id,
                        'sub_system_id' => $item['sub_system']['id'],
                        'role' => $item['sub_system']['role'] ?? $role,
                        'state' => $item['sub_system']['state'] ?? false,
                    ]);
                }
            }

            // === 建立帳號成功後，回傳資訊給SSO入口平台 ===
            $subsystemMap = [
                1 => 'SM-SI',
                2 => 'SM-EFW',
                3 => 'SM-CSM',
            ];

            if (!empty($request->items) && is_array($request->items)) {
                $parentUsername = null;
                if (!empty($request->belongs)) {
                    $parentUser = User::find($request->belongs);
                    if ($parentUser) {
                        $parentUsername = $parentUser->email;
                    }
                }

                foreach ($request->items as $item) {
                    $subId = $item['sub_system']['id'];
                    $state = $item['sub_system']['state'] ?? false;

                    if ($state && isset($subsystemMap[$subId])) {
                        $ssoPayload = [
                            'username' => $user->email,
                            'parentUsername' => $parentUsername ?? '',
                            'aliasname' => $user->name,
                            'activeSystem' => $subsystemMap[$subId],
                            'deactiveSystem' => '',
                        ];

                        Log::info("Sub User Create SSO Payload for {$subsystemMap[$subId]}", $ssoPayload);

                        // ⬇ 每個子系統都要個別呼叫一次
                        $this->SSOService->syncUserToSSO($ssoPayload);
                    }
                }
            }

            // TODO 這邊可以去抓子系統token以事先創建資料
            $this->UserService->tryLoginSubSystem($user);
            
            // === 結束 ===

            DB::commit();
            return response()->json([
                'status' => true,
                'message' => 'User Created Successfully',
                'token' => $user->api_token
            ], 200);

        } catch (\Throwable $e) {
            DB::rollback();
            return response()->json([
                'status' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $result = $this->UserService->show($id);
        $result->load('systemRole.subSystem');
        $result['pwd_last_upd'] = str($result->passwordLog()->latest()->first()->created_at ?? "");

        return response()->json(['data' => $result], 200);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $user = User::find($id);

        $user->name = $request->name;
        $user->phone = $request->phone;
        $user->contact_name = $request->contact_name;
        $user->tax_number = $request->tax_number;
        $user->group_id = $request->group_id ?? null;
        $user->save();

        if ($request->permission === '010') {
            foreach ($request->system_role as $item) {
                UserSystemRole::updateOrCreate([
                    'user_id' => $id,
                    'sub_system_id' => $item['sub_system']['id'],
                ], [
                    'state' => $item['state'],
                ]);
            }
        }

        if ($request->permission === '001') {
            foreach ($request->system_role as $item) {
                UserSystemRole::updateOrCreate([
                    'user_id' => $id,
                    'sub_system_id' => $item['sub_system']['id'],
                ], [
                    'role' => $item['role'] ?? "",
                    'state' => $item['state'] ?? false,
                ]);
            }
        }

        // TODO 這邊可以去抓子系統token以事先創建資料
        $this->UserService->tryLoginSubSystem($user);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $user = User::find($id);
        if ($user->permission === '100' || $user->permission === '010') {
            $sub_count = User::where('belongs', $user->id)->count();
            if ($sub_count > 0) {
                return response()->json([
                    'status' => false,
                    'message' => '錯誤',
                    'error' => "尚有子帳號未刪除！"
                ], 500);
            }
        }
        $this->UserService->delete($id);
    }

    // 建立測試帳號用(多筆)
    public function testUserAdd(Request $request)
    {
        // $datas = [];

        // for ($i = 12; $i <= 40; $i++) {
        //     $num = str_pad($i, 2, '0', STR_PAD_LEFT); // 格式化數字，確保為兩位數 (01, 02, ..., 50)
        //     $datas[] = [
        //         'name' => "0214-{$num}",
        //         'email' => "0214-{$num}@mail.com",
        //         'contact_name' => "0214-{$num}",
        //         'permission' => '010',
        //         'phone' => '0987654321',
        //         'tax_number' => null,
        //         'belongs' => 219,
        //         'group_id' => null,
        //         'password' => 'iredu0214',
        //     ];
        // }

        $datas = [
            [
                'name' => "chu",
                'email' => "chu@mail.isha.org.tw",
                'contact_name' => "chu",
                'permission' => '100',
                'phone' => '0987654321',
                'tax_number' => null,
                'belongs' => null,
                'group_id' => null,
                'password' => 'testtest',
            ],
            [
                'name' => "w048700",
                'email' => "w048700@mail.isha.org.tw",
                'contact_name' => "w048700",
                'permission' => '100',
                'phone' => '0987654321',
                'tax_number' => null,
                'belongs' => null,
                'group_id' => null,
                'password' => 'testtest',
            ],
            [
                'name' => "chenyi",
                'email' => "chenyi@mail.isha.org.tw",
                'contact_name' => "chenyi",
                'permission' => '100',
                'phone' => '0987654321',
                'tax_number' => null,
                'belongs' => null,
                'group_id' => null,
                'password' => 'testtest',
            ],
        ];

        DB::beginTransaction();
        try {
            foreach ($datas as $data) {
                $user = User::create([
                    'name' => $data['name'],
                    'email' => $data['email'],
                    'permission' => $data['permission'],
                    'phone' => $data['phone'],
                    'contact_name' => $data['contact_name'],
                    'tax_number' => $data['tax_number'],
                    'belongs' => $data['belongs'],
                    'group_id' => $data['group_id'],
                    'password' => Hash::make($data['password']),
                    'api_token' => Str::random(64),
                    'uuid' => Str::random(64),
                    'email_verified_at' => Carbon::now(),
                ], 201);
    
                $hasRole = false;
                $role = "";
                switch ($data['permission']) {
                    case '010':
                        $hasRole = true;
                        $role = 'sub';
                        break;
                    case '001':
                        // 需要處理選擇角色的問題
                        $hasRole = true;
                        break;
                }
    
                if ($hasRole) {
                    UserSystemRole::create([
                        'user_id' => $user->id,
                        'sub_system_id' => 2,
                        'role' => $role,
                        'state' => true,
                    ]);
                }
    
                $this->UserService->tryLoginSubSystem($user);
            }

            DB::commit();
            return response()->json([
                'status' => true,
                'message' => 'Test Users Created Successfully'
            ], 200);

        } catch (\Throwable $e) {
            DB::rollback();
            return response()->json([
                'status' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }
}
