<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Tymon\JWTAuth\Contracts\JWTSubject;
use Illuminate\Database\Eloquent\SoftDeletes;
use DateTimeInterface;
use Tymon\JWTAuth\Facades\JWTAuth;
use App\Notifications\CustomVerifyEmail;
use App\Notifications\CustomResetPassword;
use App\Services\SubSystemService;

class User extends Authenticatable implements JWTSubject, MustVerifyEmail
{
    use HasApiTokens, HasFactory, Notifiable, SoftDeletes;

    private static $system_id;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'permission',
        'contact_name',
        'phone',
        'tax_number',
        'belongs',
        'group_id',
        'api_token',
        'uuid',
        'email_verified_at',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'password' => 'hashed',
        'created_at' => 'datetime:Y-m-d H:i:s',
        'updated_at' => 'datetime:Y-m-d H:i:s',
    ];

    protected function serializeDate(DateTimeInterface $date)
    {
        return $date->format('Y-m-d\TH:i:sP');
    }

    public function sendEmailVerificationNotification()
    {
        $this->notify(new CustomVerifyEmail);
    }

    public function sendPasswordResetNotification($token)
    {
        $this->notify(new CustomResetPassword($token, $this->email));
    }

    /*
     * Get the identifier that will be stored in the subject claim of the JWT.
     *
     * @return mixed   
     */
    public function getJWTIdentifier()
    {
        return $this->getKey();
    }

    /*
     * Return a key value array, containing any custom claims to be added to the JWT.
     * 
     * @return array
     */
    public function getJWTCustomClaims()
    {
        $belongsParentData = null;

        $role = 'company';
        if($this->permission == '999') {
            $role = 'admin';
        } else {
            if(!($this->permission == '100')) {
                $belongsParentData = $this->belongsParent ? $this->belongsParent->only(['id', 'uuid', 'email', 'name', 'contact_name', 'phone', 'tax_number']) : null;
                $belongsParentData['role'] = 'company';
                $belongsParentData['contact_name'] = $belongsParentData['name'];
    
                if (!($this->permission == '010')) {
                    $systemRole = $this->systemRole->filter(function ($role) {
                        return $role->sub_system_id == self::$system_id;
                    })->values()->first();
    
                    $role = $systemRole->role;
                }
            }

            $subSystemService = app(SubSystemService::class);
            $systemState = $subSystemService->getOneUserSubSystemState($this->id);
            $systemState = $systemState->filter(function ($state) {
                return $state->id == self::$system_id;
            })->values()->first();
            
            $items = [];
            foreach ($systemState->user_plan->items as $item) {
                $items[] = [
                    'item_name' => $item->item->item,
                    'item_amount' => $item->amount,
                ];
            }
        }

        return [
            'api_token' => $this->api_token,
            'uuid' => $this->uuid,
            'email' => $this->email,
            'name' => $this->name,
            'contact_name' => $this->contact_name ?? $this->name,
            'phone' => $this->phone ?? "",
            'tax_number' => $this->tax_number ?? "",
            'role' => $role,
            'items' => $items ?? [],
            'expiry_date' => $systemState->expiry_date ?? null,
            'order_number' => $systemState->user_plan->order_number ?? null,
            'belongsParent' => $belongsParentData,
        ];
    }

    public function sub()
    {
        return $this->hasMany(User::class, 'belongs', 'id')->where('permission', '010')->orderBy('id');
    }

    public function subUser()
    {
        return $this->hasMany(User::class, 'belongs', 'id')->where('permission', '001')->orderBy('id');
    }

    public function belongsParent()
    {
        return $this->belongsTo(User::class, 'belongs', 'id');
    }

    public function group()
    {
        return $this->hasOne(AdminGroup::class, 'id', 'group_id');
    }

    public function systemRole()
    {
        return $this->hasMany(UserSystemRole::class, 'user_id', 'id');
    }

    public function jwtToken($system_id)
    {
        self::$system_id = $system_id;
        return JWTAuth::fromUser($this);
    }

    public function passwordLog()
    {
        return $this->hasMany(PasswordLog::class);
    }
}
